<script src="https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js"></script>

let selectedFile = null;
let pageRotations = {}; // { pageIndex: 90, 180, 270 }

const fileInput = document.getElementById("fileInput");
const dropZone = document.getElementById("dropZone");

const stepUpload = document.getElementById("step-upload");
const stepEditor = document.getElementById("step-editor");
const stepSuccess = document.getElementById("step-success");

const pagesContainer = document.getElementById("pagesGrid");
const applyBtn = document.getElementById("applyRotateBtn");
const downloadBtn = document.getElementById("compressBtn");

const loadingOverlay = document.getElementById("loadingOverlay");

// ----------------------------
// FILE UPLOAD
// ----------------------------
dropZone.addEventListener("click", () => fileInput.click());

fileInput.addEventListener("change", e => {
  if (!e.target.files.length) return;
  handleFile(e.target.files[0]);
});

function handleFile(file) {
  if (file.type !== "application/pdf") {
    alert("Only PDF files allowed");
    return;
  }

  selectedFile = file;
  pageRotations = {};

  stepUpload.classList.add("hidden");
  stepEditor.classList.remove("hidden");

  loadPdfPreview(file);
}

// ----------------------------
// LOAD PDF PREVIEW (PDF.js)
// ----------------------------
async function loadPdfPreview(file) {
  pagesContainer.innerHTML = "";

  const pdfjsLib = window["pdfjs-dist/build/pdf"];
  pdfjsLib.GlobalWorkerOptions.workerSrc =
    "https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js";

  const pdf = await pdfjsLib.getDocument(URL.createObjectURL(file)).promise;

  for (let i = 1; i <= pdf.numPages; i++) {
    const page = await pdf.getPage(i);
    const viewport = page.getViewport({ scale: 0.3 });

    const canvas = document.createElement("canvas");
    const ctx = canvas.getContext("2d");
    canvas.width = viewport.width;
    canvas.height = viewport.height;
    document.getElementById("pageCountBadge").textContent = pdf.numPages;

    await page.render({ canvasContext: ctx, viewport }).promise;

    const pageBox = document.createElement("div");
    pageBox.className =
      "relative border rounded-xl p-2 bg-white shadow-sm";

    pageBox.innerHTML = `
      <div class="absolute top-2 right-2 flex gap-1">
        <button onclick="rotatePage(${i}, -90)" class="rotate-btn">⟲</button>
        <button onclick="rotatePage(${i}, 90)" class="rotate-btn">⟳</button>
      </div>
    `;

    pageBox.appendChild(canvas);
    pagesContainer.appendChild(pageBox);
  }
}

// ----------------------------
// ROTATE PAGE (UI)
// ----------------------------
function rotatePage(pageIndex, angle) {
  pageRotations[pageIndex] = (pageRotations[pageIndex] || 0) + angle;

  if (pageRotations[pageIndex] >= 360) pageRotations[pageIndex] -= 360;
  if (pageRotations[pageIndex] <= -360) pageRotations[pageIndex] += 360;

  const pageEl = pagesContainer.children[pageIndex - 1].querySelector("canvas");
  pageEl.style.transform = `rotate(${pageRotations[pageIndex]}deg)`;
}

// ----------------------------
// ROTATE ALL
// ----------------------------
function rotateAll(angle) {
  for (let i = 0; i < pagesContainer.children.length; i++) {
    rotatePage(i + 1, angle);
  }
}

// ----------------------------
// APPLY ROTATION (SEND TO SERVER)
// ----------------------------
applyBtn.addEventListener("click", async () => {
  if (!selectedFile) return;

  const formData = new FormData();
  formData.append("file", selectedFile);
  formData.append("rotations", JSON.stringify(pageRotations));

  loadingOverlay.classList.remove("hidden");

  try {
    const res = await fetch("/api/pdf/rotate", {
      method: "POST",
      body: formData
    });

    if (!res.ok) throw new Error("Rotation failed");

    const blob = await res.blob();
    const url = URL.createObjectURL(blob);

    downloadBtn.href = url;
    downloadBtn.download = "PDF_NEX_ROTATED.pdf";

    stepEditor.classList.add("hidden");
    stepSuccess.classList.remove("hidden");

  } catch (err) {
    alert("Failed to rotate PDF");
    console.error(err);
  } finally {
    loadingOverlay.classList.add("hidden");
  }
});

// ----------------------------
// RESET
// ----------------------------
function resetAll() {
  location.reload();
}
